<?PHP
#
#   FILE:  UserLogin.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2002-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#


/**
* Determine if this pageload was a cross-domain one. See the comments
* in the main part of this function for referencets on this.
* @return bool TRUE for cross-domain requests, FALSE otherwise.
*/
function WasCrossDomainRequest()
{
    if (isset($_SERVER["HTTP_ORIGIN"]))
    {
        # it the HTTP_ORIGIN header was provided as part of an AJAX
        # request, extract just the domain part from it so that we can
        # check if the provided domain is one of our configured alternate
        # domains
        $OriginDomain = preg_replace(
            "%^https?://%", "", $_SERVER["HTTP_ORIGIN"]);

        if (in_array($OriginDomain, $GLOBALS["AF"]->GetAlternateDomains()))
        {
            return TRUE;
        }
    }

    return FALSE;
}

/**
* For cross-domain AJAX logins, determine the equivalent URL for the
* Referer within our primary domain so that our login javascript can
* redirect the user there.  This is necessary because the login cookie
* we've just set for the user is only valid for our primary domain.
* @return string Url in the primary domain that the user should be
*   redirected to.
*/
function GetDestinationInPrimaryDomain()
{
    # parse out the parts of the Referer
    $RefParts = parse_url($_SERVER["HTTP_REFERER"]);

    # strip off the base path
    $Path = preg_replace(
        "%^".preg_quote(ApplicationFramework::BasePath(), "%")."%",
        "", $RefParts["path"]);

    # look up the prefix for this alternate domain
    $Prefix = $GLOBALS["AF"]->GetPrefixForAlternateDomain(
        $RefParts["host"]);

    # add trailing slash if necessary
    if (substr($Prefix, -1) != "/")
    {
        $Prefix .= "/";
    }

    # construct the path in our primary domain
    return ApplicationFramework::BaseUrl().$Prefix.$Path;
}

/**
* This function is used to handle the result of the page.
* Depending on whether this page was reached by AJAX or not,
* it either sets JumpToPage or emits an AjaxMessage
* @param string $JumpToPage The page to jump to
* @param string $AjaxMessage The message to return through AJAX call
*/
function RespondToUser($JumpToPage, $AjaxMessage)
{
    # print message if reached by AJAX, otherwise set the appropriate JumpToPage
    if (ApplicationFramework::ReachedViaAjax())
    {
        $GLOBALS["AF"]->BeginAjaxResponse();
        $Response = array(
            "Status" => $AjaxMessage,
            "Redirect" => $JumpToPage
            );

        # if this was a cross-domain login, we need to figure out
        # where to send the user after they are logged in
        if (WasCrossDomainRequest())
        {
            $Response["Destination"] = GetDestinationInPrimaryDomain();
        }

        print json_encode($Response);
    }
    else
    {
        $GLOBALS["AF"]->SetJumpToPage($JumpToPage);
    }
}

# A brief digression on the same-origin policy and cross-origin
# resource sharing (CORS): Modern browsers have a LOT of feelings
# about it when javascript loaded from one domain wants to issue an
# AJAX POST against another domain. Sadly, we need to care about this
# to support AJAX logins from alternate domains (e.g., from the MSites
# plugin). Without doing the special CORS dance, the javascript
# runtime in many browsers will just fail such requests with errors
# that make basically no sense.
#
# The super-short version of the parts we care -- Browsers send an
# "Origin" header along with any cross-origin AJAX calls. Prior to
# doing a cross-origin post (e.g., to a different domain), browsers
# will "preflight" the request by sending an http OPTIONS to the
# target url. If the remote server wishes to allow the request, then
# its reply should include various Access-Control-Allow-* headers to
# indicate this. If the right headers are present, then the browser
# will follow up with a POST request. The reply to the POST must
# *also* include the same Access-Control-Allow-* headers for the
# XmlHttpRequest to complete successfully.
#
# Further-reading for the curious and/or masochistic:
#
# - https://en.wikipedia.org/wiki/Same-origin_policy
# - https://en.wikipedia.org/wiki/Cross-origin_resource_sharing
# - https://developer.mozilla.org/en-US/docs/Web/HTTP/Access_control_CORS

# if there was an origin header, then this is a cross-origin request
# and we need to set the appropriate Access-Control-Allow-* headers
if (WasCrossDomainRequest())
{
    header("Access-Control-Allow-Origin: ".$_SERVER["HTTP_ORIGIN"]);
    header("Access-Control-Allow-Headers: X-Requested-With");
    header("Access-Control-Allow-Methods: POST");
    header("Access-Control-Allow-Credentials: true");
}

# if this is a cross-domain request preflight, then all it needed were
# the headers we just generated
if ($_SERVER["REQUEST_METHOD"]=="OPTIONS")
{
    $GLOBALS["AF"]->SuppressHTMLOutput();
    return;
}

# try to log user in
$LoginResult = NULL;
if (isset($_POST["F_UserName"]) && isset($_POST["F_Password"]))
{
    $Password = "";
    $UserName = $_POST["F_UserName"];

    # if an encrypted password was sent
    if (isset($_POST["F_CryptPassword"]) && isset($_POST["UseSecure"]) )
    {
        $Password = SPTUser::DecryptPassword(
            $UserName, $_POST["F_CryptPassword"]);
    }
    # if a hashed password was sent (for backward compatibility with
    # the first version of secure login)
    elseif (isset($_POST["F_HashPassword"]) && strlen($_POST["F_HashPassword"])>0)
    {
        $Password = " ".$_POST["F_HashPassword"];
    }
    # otherwise this is a plain text password
    else
    {
        $Password = $_POST["F_Password"];
    }

    # allow plugins to override authentication by a signal
    $SignalResult = $GLOBALS["AF"]->SignalEvent(
        "EVENT_USER_AUTHENTICATION", array(
            "UserName" => $UserName,
            "Password" => $Password));

    # if the login was not forced to success or failure, proceed as normal
    if ($SignalResult === NULL)
    {
        $LoginResult = $GLOBALS["G_User"]->Login($UserName, $Password);
    }
    # if success was forced, log the user in unconditionally
    elseif ($SignalResult === TRUE)
    {
        $LoginResult = $GLOBALS["G_User"]->Login($UserName, $Password, TRUE);
    }
    # otherwise, fail the login and stop processing
    else
    {
        RespondToUser("LoginError", "Failed");
        return;
    }
}

# if login was successful
if ($LoginResult === U_OKAY)
{
    # is user account disabled?
    if ($GLOBALS["G_User"]->HasPriv(PRIV_USERDISABLED))
    {
        # log user out
        $GLOBALS["G_User"]->Logout();
        RespondToUser("LoginError", "Failed");
        return;
    }

    # signal successful user login
    $GLOBALS["AF"]->SignalEvent("EVENT_USER_LOGIN", array(
        "UserId" => $GLOBALS["G_User"]->Id(), "Password" => $Password));

    # list of pages we do not want to return to
    $DoNotReturnToPages = array(
        "Login",
        "UserLogin",
        "LoginError",
        "ForgottenPasswordComplete",
        "RequestAccount",
        "RequestAccountComplete",
        "ActivateAccount",
        "ResendAccountActivation",
        "ResetPassword",
        "ForgottenPasswordComplete",
        "EditResourceComplete"
    );

    #  if referer isn't available
    if (!isset($_POST["HTTP_REFERER"])
        && !isset($_SERVER["HTTP_REFERER"]))
    {
        # go to front page
        $ReturnPage = "Home";
    }
    else
    {
        # if we know what internal page we are returning to
        $ReturnPage = isset($_POST["HTTP_REFERER"])
            ? $_POST["HTTP_REFERER"]
            : $_SERVER["HTTP_REFERER"];
        $UnmappedReturnPage = $GLOBALS["AF"]->GetUncleanUrlForPath($ReturnPage);
        $QueryString = parse_url($UnmappedReturnPage, PHP_URL_QUERY);
        $QueryVars = ParseQueryString($QueryString);
        if (isset($QueryVars["P"]))
        {
            # go to front page if page is on "Do Not Return To" list
            if (in_array($QueryVars["P"], $DoNotReturnToPages))
            {
                $ReturnPage = "Home";
            }
            # strip out results per page if returning to advanced search
            # (to avoid overwriting user's preferences)
            elseif ($QueryVars["P"] == "AdvancedSearch")
            {
                $ReturnPage = preg_replace(
                    "/\&(amp;)?RP=[0-9]+/", "", $ReturnPage);
            }
        }
    }

    # give any hooked filters a chance to modify return page
    $SignalResult = $GLOBALS["AF"]->SignalEvent(
        "EVENT_USER_LOGIN_RETURN",
        array("ReturnPage" => $ReturnPage));
    $ReturnPage = $SignalResult["ReturnPage"];

    # set destination to return to after login
    RespondToUser($ReturnPage, "Success");
    return;
}
elseif ($LoginResult == U_NOTACTIVATED)
{
    # go to "needs activation" page
    $ReturnPage = "index.php?P=UserNotActivated&UN=".urlencode($UserName);
    RespondToUser($ReturnPage, "Redirect");
    return;
}
elseif (isset($Password)
        && (preg_match("/^[0-9A-F]{6}([0-9A-F]{4})?$/", $Password) == 1))
{
    # login failed, but password looks like it was a reset or an
    # activation code

    # see if the provided username or email exists
    $UFact = new CWUserFactory();
    if ($UFact->UserNameExists($UserName) ||
        $UFact->EMailAddressIsInUse($UserName) )
    {
        $TargetUser = new SPTUser($UserName);

        # if the account is already activated
        if ($TargetUser->IsActivated())
        {
            # see if the code provided was a valid password reset code
            if ($TargetUser->IsResetCodeGood($Password))
            {
                # jump to the reset page if so
                $ReturnPage = "index.php?P=ResetPassword&UN=".$UserName."&RC=".$Password;
                RespondToUser($ReturnPage, "Redirect");
                return;
            }
        }
        # otherwise, see if the code provided was a valid activation code
        elseif ($TargetUser->IsActivationCodeGood($Password))
        {
            # jump to the activation page if so
            $ReturnPage = "index.php?P=ActivateAccount&"."UN=".$UserName."&".
                "AC=".$Password;
            RespondToUser($ReturnPage, "Redirect");
            return;
        }
    }
}

# fail any other logins
RespondToUser("LoginError", "Failed");
