<?PHP
class Image_Test extends PHPUnit\Framework\TestCase
{
	protected static $TempDir;
	/**
	* Create sandbox folder for storing new images.
	*/
	static function setUpBeforeClass()
	{
		self::$TempDir = "tmp/Image--Test-".getmypid();
		mkdir(self::$TempDir);
	}

	/**
	* Destroy sandbox folder and contents.
	*/
	static function tearDownAfterClass()
	{
		$Files = glob(self::$TempDir."/*");
		foreach ($Files as $File)
		{
			if (is_file($File))
			{
				unlink($File);
			}
		}
		rmdir(self::$TempDir);
	}

	function testAll()
	{
		# construct Image with invalid file
		$BadImage = new Image("abc123");
		$this->assertEquals($BadImage->Status(), AI_FILEUNREADABLE);

		# Test SaveAs() and Status() with no file changes
		$ImageJpg = new Image("lib/ScoutLib/tests/files/TestImage--600x400.jpg");
		$ImageJpg->SaveAs(self::$TempDir."/TestImage2--600x400.jpg");
		$this->assertEquals($ImageJpg->Status(), AI_OKAY, "Testing SaveAs() for JPG");

		# Test Xsize() and Ysize() of 600x400 JPG image.
		$this->assertEquals($ImageJpg->XSize(), 600,
			"Testing Xsize() of 600px width image");
		$this->assertEquals($ImageJpg->YSize(), 400,
			"Testing Ysize() of 400px height image");

		# Test ScaleTo() without maintaining aspect ratio and converting
		# image to gif
		$ImageJpg->ScaleTo(100, 200);
		$ImageJpg->SaveAs(self::$TempDir."/TestImage--100x200.gif",
			Image::IMGTYPE_GIF);

		$ImageGif = new Image(self::$TempDir."/TestImage--100x200.gif");
		$this->assertEquals($ImageGif->Status(), AI_OKAY, "Testing SaveAs() for GIF");
		$this->assertEquals($ImageGif->XSize(), 100,
			"Testing ScaleTo() of 100px width image");
		$this->assertEquals($ImageGif->YSize(), 200,
			"Testing ScaleTo() of 200px height image");

		# Test ScaleTo() with maintaining aspect ratio and converting image to png
		# Because the aspect ratio is maintained, we should expect a 200x400 image.
		$ImageGif->ScaleTo(200, 300, TRUE);
		$ImageGif->SaveAs(self::$TempDir."/TestImage--200x400.png",
			Image::IMGTYPE_PNG);

		$ImagePng = new Image(self::$TempDir."/TestImage--200x400.png");
		$this->assertEquals($ImagePng->Status(), AI_OKAY, "Testing SaveAs() for PNG");
		$this->assertEquals($ImagePng->XSize(), 200,
			"Testing ScaleTo() of 200px width image, with aspect ratio maintained");
		$this->assertEquals($ImagePng->YSize(), 400,
			"Testing ScaleTo() of 400px height image, with aspect ratio maintained");

		# Test CropTo() and converting image to bmp
		$ImagePng->CropTo(100, 150);
		$ImagePng->SaveAs(self::$TempDir."/TestImage--100x150.bmp");

		$ImageBmp = new Image(self::$TempDir."/TestImage--100x150.bmp");
		$this->assertEquals($ImageBmp->Status(), AI_OKAY, "Testing SaveAs() for BMP");
		$this->assertEquals($ImageBmp->XSize(), 100,
			"Testing CropTo() of 100px width image");
		$this->assertEquals($ImageBmp->YSize(), 150,
			"Testing CropTo() of 150px height image");

		# Test Type()
		$this->assertEquals($ImageJpg->Type(), Image::IMGTYPE_JPEG, "Testing Type()");
		$this->assertEquals($ImageJpg->Type(self::$TempDir."/TestImage2--600x400.jpg"),
			Image::IMGTYPE_JPEG, "Testing Type() with 'TestImage2--600x400.jpg'");
		$this->assertEquals($ImageJpg->Type(self::$TempDir."/TestImage--100x200.gif"),
			Image::IMGTYPE_GIF, "Testing Type() with 'TestImage--100x200.gif'");
		$this->assertEquals($ImageJpg->Type(self::$TempDir."/TestImage--100x150.bmp"),
			Image::IMGTYPE_BMP, "Testing Type() with 'TestImage--100x150.bmp'");
		$this->assertEquals($ImageJpg->Type(self::$TempDir."/TestImage--200x400.png"),
			Image::IMGTYPE_PNG, "Testing Type() with 'TestImage--200x400.png'");
		$this->assertEquals($ImageJpg->Type("FakeFile.xyz"), Image::IMGTYPE_UNKNOWN,
			"Testing Type() with 'FakeFile.xyz'");

		# Test MimeType()
		$this->assertEquals($ImageJpg->Mimetype(), "image/jpeg",
			"Testing Mimetype() for jpg image");
		$this->assertEquals($ImageGif->Mimetype(), "image/gif",
			"Testing Mimetype() for gif image");
		$this->assertEquals($ImageBmp->Mimetype(), "image/bmp",
			"Testing Mimetype() for bmp image");
		$this->assertEquals($ImagePng->Mimetype(), "image/png",
			"Testing Mimetype() for png image");

		# Test Extension()
		$this->assertEquals($ImageJpg->Extension(), "jpg", "Testing Extension()");

		# Test ExtensionForType()
		$this->assertEquals(Image::ExtensionForType(Image::IMGTYPE_JPEG), "jpg",
			"Testing ExtensionForType() with jpg");
		$this->assertEquals(Image::ExtensionForType(Image::IMGTYPE_GIF), "gif",
			"Testing ExtensionForType() with gif");
		$this->assertEquals(Image::ExtensionForType(Image::IMGTYPE_BMP), "bmp",
			"Testing ExtensionForType() with bmp");
		$this->assertEquals(Image::ExtensionForType(Image::IMGTYPE_PNG), "png",
			"Testing ExtensionForType() with png");
		$this->assertEquals(Image::ExtensionForType(123), NULL,
			"Testing ExtensionForType() with 123");

		# Test JpegQuality()
		$this->assertEquals($ImageJpg->JpegQuality(50), 50,
			"Testing JpegQuality() with 80");

		# Test SupportedFormats()
		# we assume that jpeg, gif, bmp, and png images will all be supported
		$this->assertEquals(Image::SupportedFormats(), Image::IMGTYPE_JPEG |
			Image::IMGTYPE_GIF | Image::IMGTYPE_PNG | Image::IMGTYPE_BMP,
			"Testing SupportedFormats()");

		# Test SupportedFormatNames()
		# we assume that jpeg, gif, bmp, and png images will all be supported
		$this->assertContains("JPG", Image::SupportedFormatNames(),
			"Testing SupportedFormatNames contains 'JPG'");
		$this->assertContains("GIF", Image::SupportedFormatNames(),
			"Testing SupportedFormatNames contains 'GIF'");
		$this->assertContains("BMP", Image::SupportedFormatNames(),
			"Testing SupportedFormatNames contains 'BMP'");
		$this->assertContains("PNG", Image::SupportedFormatNames(),
			"Testing SupportedFormatNames contains 'PNG'");

		# Test SaveAs() error checking
		$ImageJpg->SaveAs(self::$TempDir."/TestImage--100x200.gif",
			"xyz");
		$this->assertEquals($ImageJpg->Status(), AI_UNSUPPORTEDFORMAT,
			"Testing SaveAs() with AI_UNSUPPORTEDFORMAT");
	}
}
