<?PHP
#
#   FILE:  PrivilegeEditingUI.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2015-2016 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* User interface element for editing PrivilegeSets.  The enclosing form
* must have the class "priv-form".
*/
class PrivilegeEditingUI
{
    /**
    * Constructor for privilege editing UI.
    * @param int $Schemas SchemaId or array of SchemaIds that will be
    *       used for any fields referenced in privilege conditions
    *       (OPTIONAL, defaults to all Schemas)
    * @param array $MetadataFields Array of metadata field objects (keyed by
    *       FieldId) listing fields that should be displayed.  If this
    *       argument is specified, the $SchemaIds argument must be
    *       NULL.
    */
    public function __construct($Schemas = NULL, $MetadataFields = array())
    {
        $this->Fields = array();

        # if Schemas was NULL, use all schemas
        if ($Schemas === NULL)
        {
            $Schemas = MetadataSchema::GetAllSchemas();
        }
        else
        {
            # ensure incoming value is an array
            if (!is_array($Schemas))
            {
                $Schemas = array($Schemas);
            }

            # if we have an array of ints, convert to an array of MetadataSchema objects
            if (is_numeric(reset($Schemas)))
            {
                $NewSchemas = array();
                foreach ($Schemas as $SchemaId)
                {
                    $NewSchemas[$SchemaId] = new MetadataSchema($SchemaId);
                }
                $Schemas = $NewSchemas;
            }
        }

        # ensure incoming value is an array
        if (!is_array($MetadataFields))
        {
            $MetadataFields = array($MetadataFields);
        }

        # if we have an array of ints, convert to an array of MetadataField objects
        if (is_numeric(reset($MetadataFields)))
        {
            $NewMetadataFields = array();
            foreach ($MetadataFields as $FieldId)
            {
                $NewMetadataFields[$FieldId] = new MetadataField($FieldId);
            }
            $MetadataFields = $NewMetadataFields;
        }

        # types we support, in the order they should be displayed
        $SupportedFieldTypesInOrder = array(
            MetadataSchema::MDFTYPE_USER,
            MetadataSchema::MDFTYPE_FLAG,
            MetadataSchema::MDFTYPE_OPTION,
            MetadataSchema::MDFTYPE_TIMESTAMP,
            MetadataSchema::MDFTYPE_NUMBER);

        # add all requested Schemas
        foreach ($Schemas as $SchemaId => $Schema)
        {
            # iterate over the supported types so that fields are
            # returned grouped by type
            foreach ($SupportedFieldTypesInOrder as $Type)
            {
                $this->Fields += $Schema->GetFields($Type);
            }
        }

        # and add requested fields
        foreach ($MetadataFields as $FieldId => $Field)
        {
            if (!in_array($Field->Type(), $SupportedFieldTypesInOrder))
            {
                throw new Exception(
                    "Field ".$Field->Name()." (Id=".$Field->Id().")"
                    ." is invalid for PrivilegeEditing -- ".$Field->TypeAsName()
                    ." fields are not supported.");
            }
            $this->Fields[$FieldId] = $Field;
        }
    }

    /**
    * Display interface for editing specified privilege set.
    * @param string $Identifier Alphanumeric identifier for this privilege set.
    * @param object $PrivilegeSet Current values for privilege set.
    * @param bool $IsNested For recursion only - DO NOT USE.
    */
    public function DisplaySet($Identifier, PrivilegeSet $PrivilegeSet,
            $IsNested = FALSE)
    {
        # include needed JavaScript
        $GLOBALS["AF"]->RequireUIFile(__CLASS__.".js");

        # build form field prefix
        $FFPrefix = "F_Priv_".$Identifier."_";

        # retrieve privilege info as an array
        $PrivilegeInfo = $PrivilegeSet->GetPrivilegeInfo();

        # add "Top-Level Logic" option list if we are at the top of the hierarchy
        if (!$IsNested)
        {
            $Logic = $PrivilegeInfo["Logic"];

            print '<div class="priv-set">'
                .'<fieldset class="priv-fieldset priv-logic">'
                .'<label for="'.$FFPrefix.'Logic">Top-Level Logic:</label>';

            $OptList = new HtmlOptionList($FFPrefix."Logic",
                    array("AND" => "AND", "OR" => "OR"), $Logic);
            $OptList->PrintHtml();

            print '</fieldset>';
        }

        # discard logic so that the rest of the array can be iterated over
        unset($PrivilegeInfo["Logic"]);

        # if there are no conditions set
        if (count($PrivilegeInfo) == 0)
        {
            # print out message indicating no conditions yet set
            print("<i>(no requirements)</i><br/>");
        }
        else
        {
            # print out each condition
            foreach ($PrivilegeInfo as $Condition)
            {
                ?>
                <fieldset class="priv-fieldset cw-peui-fieldset">
                <?PHP

                # if condition tests against a user privilege
                if (is_numeric($Condition))
                {
                    $this->DisplaySubjectField($FFPrefix, "current_user");
                    $this->DisplayOperatorField($FFPrefix);
                    $this->DisplayValueField($FFPrefix, $Condition);
                }
                # else if condition tests against a metadata field value
                else if (is_array($Condition))
                {
                    $this->DisplaySubjectField($FFPrefix, $Condition["FieldId"]);
                    $this->DisplayOperatorField($FFPrefix, $Condition["Operator"]);

                    try
                    {
                        $Field = new MetadataField($Condition["FieldId"]);
                    }
                    catch (Exception $e)
                    {
                        # do nothing here, but we'd like to continue
                    }

                    if (isset($Field) && $Field->Type() == MetadataSchema::MDFTYPE_OPTION)
                    {
                        # Option fields use the selector menu, rather than a
                        #       form field.
                        # Values are ControlledName Ids, prefixed with a "C"
                        #       to distinguish them from privilge flag numbers.
                        $this->DisplayValueField($FFPrefix,
                                "C".$Condition["Value"], "NULL");
                    }
                    else
                    {
                        $this->DisplayValueField($FFPrefix, NULL, $Condition["Value"]);
                    }
                }
                # else if condition is a privilege subset
                else if ($Condition instanceof PrivilegeSet)
                {
                    $this->DisplaySubjectField($FFPrefix, "set_entry");
                    $this->DisplayOperatorField($FFPrefix);
                    $this->DisplayValueField($FFPrefix,
                            $Condition->AllRequired() ? "AND" : "OR");

                    # end the fieldset for the set entry row
                    ?></fieldset><?PHP

                    # print the nested fields
                    $this->DisplaySet($Identifier, $Condition, TRUE);

                    # begin a new fieldset for the set exit row
                    ?><fieldset class="priv-fieldset cw-peui-fieldset"><?PHP

                    $this->DisplaySubjectField($FFPrefix, "set_exit");
                    $this->DisplayOperatorField($FFPrefix);
                    $this->DisplayValueField($FFPrefix);
                }

                ?></fieldset><?PHP
            }
        }

        # if we are at the top level
        if (!$IsNested)
        {
            $NumBlankRows = 6;

            # print a number of blank rows to be used if JavaScript is disabled
            for ($Index = 0;  $Index < $NumBlankRows;  $Index++)
            {
                ?><fieldset class="priv-fieldset cw-peui-fieldset priv-extra"><?PHP
                $this->DisplaySubjectField($FFPrefix);
                $this->DisplayOperatorField($FFPrefix);
                $this->DisplayValueField($FFPrefix);
                ?></fieldset><?PHP
            }

            # print a blank row for cloning within JavaScript
            ?><fieldset class="priv-fieldset cw-peui-fieldset priv-js-clone_target"><?PHP
            $this->DisplaySubjectField($FFPrefix);
            $this->DisplayOperatorField($FFPrefix);
            $this->DisplayValueField($FFPrefix);
            ?></fieldset><?PHP

            # print the button to add a new row when using JavaScript
            ?><button class="cw-button cw-button-elegant cw-button-constrained
                    priv-js-add">Add Condition</button>
            <?PHP

            # print the closing div for the set
            ?></div><?PHP
        }
    }

    /**
    * Construct new privilege sets from available form ($_POST) data.
    * @return array Returns an array of PrivilegeSet objects.
    */
    public function GetPrivilegeSetsFromForm()
    {
        # for each form field
        $Sets = array();
        $Logics = array();
        foreach ($_POST as $FieldName => $FieldValue)
        {
            # if field looks like privilege set data
            if (preg_match("/^F_Priv_/", $FieldName))
            {
                # extract identifier from field name
                $Pieces = explode("_", $FieldName);
                $Identifier = $Pieces[2];

                # if field looks like privilege set top-level logic
                if (preg_match("/_Logic\$/", $FieldName))
                {
                    # save logic for later use
                    $Logics[$Identifier] = $FieldValue;
                }
                else
                {
                    # retrieve privilege set from field
                    $Sets[$Identifier] =
                            $this->ExtractPrivilegeSetFromFormData($FieldValue);
                }
            }
        }

        # for each top-level logic found
        foreach ($Logics as $Identifier => $Logic)
        {
            # if no corresponding privilege set was found
            if (!isset($Sets[$Identifier]))
            {
                # load empty set for this identifier
                $Sets[$Identifier] = new PrivilegeSet();
            }

            # set logic in corresponding privilege set
            $Sets[$Identifier]->AllRequired(($Logic == "AND") ? TRUE : FALSE);
        }

        # return any privilege sets found to caller
        return $Sets;
    }

    /**
    * Retrieve privilege set from specified form ($_POST) data fields.
    * @param string $Identifier Identifier of privilege set to return.
    * @return object Privilege set or FALSE if no privilege set form data
    *       found with the specified identifier.
    */
    public function GetPrivilegeSetFromForm($Identifier)
    {
        $Sets = $this->GetPrivilegeSetsFromForm();
        return isset($Sets[$Identifier]) ? $Sets[$Identifier] : FALSE;
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    private $Fields;
    private $OptionValues;

    /**
    * Print HTML select element used to decide the subject of a condition
    * (where conditions are comprised of 'subject', 'verb', and 'object',
    * and subject can be a metadata field, Current User, etc).
    * @param string $FFPrefix Prefix to use for form fields.
    * @param mixed $Selected The value to select. (OPTIONAL)
    */
    private function DisplaySubjectField($FFPrefix, $Selected=NULL)
    {
        # construct the list of options to present for this field
        # present all the metadata fields using their FieldIds
        $ValidSelections = array_keys($this->Fields);

        # represent other elements as strings, which need to match the
        #   values in GetPrivilegeSetFromFormData()
        $ValidSelections[] = "current_user";
        $ValidSelections[] = "set_entry";
        $ValidSelections[] = "set_exit";

        $AllSchemas = MetadataSchema::GetAllSchemas();

        # construct 2d array of our fields keyed by schemaid
        $Fields = array();
        foreach ($this->Fields as $Id => $Field)
        {
            $Fields[$Field->SchemaId()][$Id]= $Field;
        }

        # build up the list options to included
        $Options = array();
        $OptionCSS = array();

        foreach ($Fields as $ScId => $ScFields)
        {
            $FieldOptions = array();
            foreach ($ScFields as $Id => $Field)
            {
                $SafeClassType = strtolower($Field->TypeAsName());
                $FieldOptions[$Id] = "[".$AllSchemas[$ScId]->AbbreviatedName()."] "
                        .$Field->GetDisplayName();
                $OptionCSS[$Id] = "priv priv-option priv-field-subject "
                        ."priv-type-".$SafeClassType."_field";
            }

            $OptLabel = $AllSchemas[$ScId]->Name();
            $Options[$OptLabel] = $FieldOptions;
        }

        # add Current User entry
        $Options["current_user"] = "Current User";
        $OptionCSS["current_user"] = "priv priv-option priv-field-subject "
                ."priv-type-privilege";

        # add subgroup begin marker
        $Options["set_entry"] = "(";
        $OptionCSS["set_entry"] = "priv priv-option priv-field-subject "
                ."priv-type-set_entry";

        # add subgroup end marker
        $Options["set_exit"] = ")";
        $OptionCSS["set_exit"] = "priv priv-option priv-field-subject "
                ."priv-type-set_exit";

        # check if the data we were given contains an invalid field,
        # and if so complain
        if (!is_null($Selected) && !in_array($Selected, $ValidSelections, TRUE))
        {
            $Options[$Selected] = "INVALID FIELD";
            $OptionCSS[$Selected] = "priv priv-option priv-field-subject "
                    ."priv-type-user_field";
        }


        $OptionList = new HtmlOptionList($FFPrefix."[]", $Options, $Selected);
        $OptionList->ClassForOptions($OptionCSS);
        $OptionList->ClassForList(
                "priv priv-field priv-select priv-field-subject priv-type-user_field "
                ."priv-type-flag_field priv-type-option_field priv-type-timestamp_field "
                ."priv-type-date_field priv-type-number_field priv-type-set_entry "
                ."priv-type-set_exit");

        $OptionList->PrintHtml();
    }

    /**
    * Print the form field for the operator field.
    * @param string $FFPrefix Prefix to use for form fields.
    * @param mixed $Selected The value to select. (OPTIONAL)
    */
    private function DisplayOperatorField($FFPrefix, $Selected=NULL)
    {
        $Options = array();
        $OptionCSS = array();
        $CommonStyles = "priv priv-option priv-field-operator";

        # use css styles on each option to indicate which fields it applies to

        # equal and not equal work for User, Flag, Option, and Number fields
        foreach (array("==", "!=") as $Op)
        {
            $Options[$Op] = $Op;
            $OptionCSS[$Op] = $CommonStyles." priv-type-user_field "
                    ."priv-type-flag_field priv-type-option_field priv-type-number_field";
        }

        # less and greater work for Timestamp, Date, and Number fields
        foreach (array("<", ">") as $Op)
        {
            $Options[$Op] = $Op;
            $OptionCSS[$Op] = $CommonStyles
                ." priv-type-timestamp_field priv-type-date_field priv-type-number_field";
        }

        $OptionList = new HtmlOptionList($FFPrefix."[]", $Options, $Selected);
        $OptionList->ClassForOptions($OptionCSS);
        $OptionList->ClassForList(
            "priv priv-field priv-select priv-field-operator priv-type-user_field "
            ."priv-type-flag_field priv-type-option_field priv-type-timestamp_field "
            ."priv-type-date_field priv-type-number_field");

        $OptionList->PrintHtml();
    }

    /**
    * Print the form fields for the value field.
    * @param string $FFPrefix Prefix to use for form fields.
    * @param mixed $Selected The value to select for the select box. (OPTIONAL)
    * @param mixed $Value The existing value for the input box. (OPTIONAL)
    */
    private function DisplayValueField($FFPrefix, $Selected=NULL, $Value=NULL)
    {
        $this->PrintPrivilegeValueSelectorField($FFPrefix, $Selected);
        $this->PrintPrivilegeValueInputField($FFPrefix, $Value);
    }

    /**
    * Construct a new privilege set from the given array of form data.
    * @param array $FormData An array of form data coming from elements
    *   generated by PrintPrivilegeFields().  For example, if
    *   PrintPrivilegeFields() was called on the previous page with
    *   PrivilegeType = "ViewingPrivileges", you should you should pass
    *   $_POST["F_ViewingPrivileges"] in here.  This variable will be
    *   modified by reference.
    * @return Returns a PrivilegeSet object.
    * @throws Exception If invalid data is given.
    */
    private function ExtractPrivilegeSetFromFormData(array &$FormData)
    {
        $NewPrivilegeSet = new PrivilegeSet();
        $Privileges = $this->GetPrivileges();
        $SupportedOperators = array("==", "!=", "<", ">");

        while (count($FormData))
        {
            # extract the form fields
            $SubjectField = array_shift($FormData);
            $OperatorField = array_shift($FormData);
            $ValueSelectField = array_shift($FormData);
            $ValueInputField = array_shift($FormData);

            # privilege condition
            if ($SubjectField == "current_user")
            {
                # invalid privilege ID
                if (!isset($Privileges[$ValueSelectField])
                        || is_null($ValueSelectField))
                {
                    throw new Exception("Invalid privilege (".$ValueSelectField.")");
                }

                $NewPrivilegeSet->AddPrivilege($ValueSelectField);
            }

            # metadata field condition
            else if (is_numeric($SubjectField))
            {
                # invalid field ID
                if (!isset($this->Fields[$SubjectField]) || is_null($SubjectField))
                {
                    throw new Exception("Invalid or unsupported field ("
                            .$SubjectField.")");
                }

                # invalid operator
                if (!in_array($OperatorField, $SupportedOperators))
                {
                    throw new Exception("Invalid or unsupported operator ("
                            .$OperatorField.")");
                }

                $MetadataField = $this->Fields[$SubjectField];

                switch ($MetadataField->Type())
                {
                    case MetadataSchema::MDFTYPE_USER:
                        $Value = NULL;
                        break;

                    case MetadataSchema::MDFTYPE_FLAG:
                        $Value = 1;
                        break;

                    case MetadataSchema::MDFTYPE_TIMESTAMP:
                    case MetadataSchema::MDFTYPE_DATE:
                    case MetadataSchema::MDFTYPE_NUMBER:
                        $Value = $ValueInputField;
                        break;

                    case MetadataSchema::MDFTYPE_OPTION:
                        # strip the "C" prefix used to distinguish controlled
                        #       names from priv flags
                        $Value = intval( substr( $ValueSelectField, 1));
                        break;

                    default:
                        $Value = NULL;
                        break;
                }

                $NewPrivilegeSet->AddCondition($MetadataField, $Value, $OperatorField);
            }

            # entering a nested privilege set
            else if ($SubjectField == "set_entry")
            {
                # the logic is invalid
                if ($ValueSelectField != "AND" && $ValueSelectField != "OR")
                {
                    throw new Exception("Invalid privilege set logic ("
                            .$ValueSelectField.")");
                }

                $NestedPrivilegeSet = $this->ExtractPrivilegeSetFromFormData($FormData);

                # only add the nested privilege set if it's not empty. use 1
                # because the logic is in the privilege info array
                if (count($NestedPrivilegeSet->GetPrivilegeInfo()) > 1)
                {
                    $NestedPrivilegeSet->AllRequired($ValueSelectField == "AND");
                    $NewPrivilegeSet->AddSet($NestedPrivilegeSet);
                }
            }

            # exiting a privilege set
            else if ($SubjectField == "set_exit")
            {
                break;
            }

            # unknown condition type
            else
            {
                throw new Exception("Unknown condition type: ".$SubjectField);
            }
        }

        return $NewPrivilegeSet;
    }

    /**
    * Print a select box for the value field.
    * @param string $FFPrefix Prefix to use for form fields.
    * @param mixed $Selected The value to select. (OPTIONAL)
    */
    private function PrintPrivilegeValueSelectorField($FFPrefix, $Selected=NULL)
    {
        # build up the list of options
        $Options = array();
        $OptionCSS = array();
        $OptionData = array();

        # add entries for each user privilege flag
        $Privileges = $this->GetPrivileges();
        foreach ($Privileges as $Id => $Privilege)
        {
            $Options[$Id] = $Privilege->Name();
            $OptionCSS[$Id] = "priv priv-option priv-field-value priv-type-privilege";
        }

        $OptionValues = $this->GetOptionValuesForPrivset();
        foreach ($OptionValues as $FieldId => $Values)
        {
            foreach ($Values as $CNId => $CName)
            {
                $Options["C".$CNId] = $CName;
                $OptionCSS["C".$CNId] = "priv priv-option priv-field-value "
                        ."priv-type-option_field";
                $OptionData["C".$CNId]["field-id"] = $FieldId;
            }
        }

        $Options["AND"] = "AND";
        $OptionCSS["AND"] = "priv priv-option priv-field-value priv-type-set_entry";

        $Options["OR"] = "OR";
        $OptionCSS["OR"] = "priv priv-option priv-field-value priv-type-set_entry";

        $OptionList = new HtmlOptionList($FFPrefix."[]", $Options, $Selected);
        $OptionList->ClassForOptions($OptionCSS);
        $OptionList->DataForOptions($OptionData);
        $OptionList->ClassForList(
            "priv priv-field priv-select priv-field-value priv-type-option_field "
            ."priv-type-privilege priv-type-set_entry priv-type-set_exit");

        $OptionList->PrintHtml();
    }

    /**
    * Print an input box for the value field.
    * @param string $FFPrefix Prefix to use for form fields.
    * @param mixed $Value The existing value. (OPTIONAL)
    */
    private function PrintPrivilegeValueInputField($FFPrefix, $Value=NULL)
    {
        $SafeValue = defaulthtmlentities($Value);
    ?>
      <input name="<?PHP print $FFPrefix; ?>[]"
             type="text"
             class="priv priv-field priv-input priv-field-value
                    priv-type-timestamp_field priv-type-date_field
                    priv-type-number_field"
             value="<?PHP print $SafeValue; ?>" />
    <?PHP
    }

    /**
    * Get the list of option values allowed for privilege sets
    * @return array Values in an array of arrays, with the outer array having
    *       FieldName for the index, and the inner arrays having OptionId
    *       for the index.
    */
    private function GetOptionValuesForPrivset()
    {
        if (!isset($this->OptionValues))
        {
            $this->OptionValues = array();

            foreach ($this->Fields as $FieldId => $Field)
            {
                if ($Field->Type() == MetadataSchema::MDFTYPE_OPTION)
                {
                    $this->OptionValues[$Field->Id()] = $Field->GetPossibleValues();
                }
            }
        }

        return $this->OptionValues;
    }

    /**
    * Get the list of privileges.
    * @return Returns an array of all privileges.
    */
    private function GetPrivileges()
    {
        static $Privileges;

        if (!isset($Privileges))
        {
            $PrivilegeFactory = new PrivilegeFactory();
            $Privileges = $PrivilegeFactory->GetPrivileges();
        }

        return $Privileges;
    }
}