<?PHP
#
#   FILE:  BarChart.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Class for generating and displaying a bar chart.
*/
class BarChart extends Chart_Base
{
    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Class constructor.
    * @param array $Data Data for chart. Keys are X values for each
    *   bar (either a category name, a unix timestamp, or any format
    *   that strtotime can parse). Values are ints for charts with
    *   only one set of bars or associative arrays where the keys give
    *   bar names and the values give bar heights for charts with
    *   multiple bars.
    */
    public function __construct($Data)
    {
        # if data provided is an array of arrays, then just copy it in
        if (is_array(reset($Data)))
        {
            $this->Data = $Data;
        }
        # otherwise, normalize to array of arrays format
        else
        {
            $this->SingleCategory = TRUE;
            $this->Stacked = TRUE;
            $this->LegendPosition = static::LEGEND_NONE;

            $this->Data = [];
            foreach ($Data as $Name => $Val)
            {
                $this->Data[$Name] = [$Name => $Val];
            }
        }
    }

    /**
    * Get/set the axis type of a bar chart (default is AXIS_CATEGORY).
    * @param string $NewValue Axis type as a BarChart::AXIS_
    *   constant. Allowed values are AXIS_CATEGORY for categorical
    *   charts or one of AXIS_TIME_{DAILY,WEEKLY,MONTHLY,YEARLY} for
    *   time series plotting.
    * @return mixed Current AxisType.
    * @throws Exception If an invalid axis type is supplied.
    */
    public function AxisType($NewValue=NULL)
    {
        if (func_num_args()>0)
        {
            $ValidTypes = [
                self::AXIS_CATEGORY,
                self::AXIS_TIME_DAILY,
                self::AXIS_TIME_WEEKLY,
                self::AXIS_TIME_MONTHLY,
                self::AXIS_TIME_YEARLY,
            ];

            # toss exception if the given type is not valid
            if (!in_array($NewValue, $ValidTypes))
            {
                throw new Exception("Invalid axis type for bar charts: ".$NewValue);
            }

            $this->AxisType = $NewValue;
        }

        return $this->AxisType;
    }

    /**
    * Get/set the Y axis label for a bar chart.
    * @param string $NewValue Label to use.
    * @return string Current YLabel.
    */
    public function YLabel($NewValue=NULL)
    {
        if (func_num_args()>0)
        {
            $this->YLabel = $NewValue;
        }

        return $this->YLabel;
    }

    /**
    * Get/set bar width as a percentage of the distance between ticks.
    * @param int|null $NewValue Updated bar width or NULL to use the C3 default.
    * @return mixed Current Barwidth setting.
    * @throws Exception If an invalid bar width is supplied.
    */
    public function BarWidth($NewValue=NULL)
    {
        if (func_num_args()>0)
        {
            if (!is_null($NewValue) &&
                $NewValue <= 0 || $NewValue > 100)
            {
                throw new Exception("Invalid bar width: ".$NewValue);
            }
            $this->BarWidth = $NewValue;
        }
        return $this->BarWidth;
    }

    /**
    * Enable/Disable zooming for this chart.
    * @param string $NewValue TRUE to enable zooming.
    * @return bool Current Zoom setting.
    * @throws Exception If an invalid value is supplied.
    */
    public function Zoom($NewValue=NULL)
    {
        if (func_num_args()>0)
        {
            if (!is_bool($NewValue))
            {
                throw new Exception("Invalid Zoom setting -- must be boolean");
            }
            $this->Zoom = $NewValue;
        }
        return $this->Zoom;
    }

    /**
    * Get/Set bar stacking setting.
    * @param bool $NewValue TRUE to generate a stacked chart.
    * @return bool Current stacking setting.
    * @throws Exception If an invalid value is supplied.
    */
    public function Stacked($NewValue=NULL)
    {
        if (func_num_args()>0)
        {
            if (!is_bool($NewValue))
            {
                throw new Exception("Invalid Stacked setting -- must be boolean");
            }

            $this->Stacked = $NewValue;
        }

        return $this->Stacked;
    }

    /**
    * Get/Set horizontal display setting
    * @param bool $NewValue TRUE to generate a horizontal bar chart.
    * @return bool Current horizontal setting.
    * @throws Exception If an invalid value is supplied.
    */
    public function Horizontal($NewValue=NULL)
    {
        if (func_num_args()>0)
        {
            if (!is_bool($NewValue))
            {
                throw new Exception("Invalid Horizontal setting -- must be boolean");
            }
            $this->Horizontal = TRUE;
        }

        return $this->Horizontal;
    }

    /**
    * Enable/disable display of grid lines.
    * @param bool $NewValue TRUE to show grid lines.
    * @return bool Current grid line sitting.
    * @throws Exception If an invalid value is supplied.
    */
    public function Gridlines($NewValue=NULL)
    {
        if (func_num_args()>0)
        {
            if (!is_bool($NewValue))
            {
                throw new Exception("Invalid Gridlines setting -- must be boolean");
            }
            $this->Gridlines = $NewValue;
        }

        return $this->Gridlines;
    }

    /**
    * Enable/disable display of category labels along the X axis on
    * categorical charts (by default, they are shown).
    * @param bool $NewValue TRUE to show category labels.
    * @return bool Current category labels setting.
    * @throws Exception If an invalid value is supplied.
    */
    public function ShowCategoryLabels($NewValue=NULL)
    {
        if (func_num_args()>0)
        {
            if (!is_bool($NewValue))
            {
                throw new Exception(
                    "Invalid ShowCategoryLabels setting -- must be boolean");
            }
            $this->ShowCategoryLabels = $NewValue;
        }

        return $this->ShowCategoryLabels;
    }

    # axis types
    const AXIS_CATEGORY = "category";
    const AXIS_TIME_DAILY = "daily";
    const AXIS_TIME_WEEKLY = "weekly";
    const AXIS_TIME_MONTHLY = "monthly";
    const AXIS_TIME_YEARLY = "yearly";

    # ---- PRIVATE INTERFACE --------------------------------------------------

    /**
    * Prepare data for plotting.
    * @see ChartBase::PrepareData().
    */
    protected function PrepareData()
    {
        # input data is
        #  [ CatNameOrTimestamp => [Data1 => Val, Data2 => Val, ...], ... ]
        #
        # and the format that C3 expects is
        # [ "Data1", Val, Val, ... ]
        # [ "Data2", Val, Val, ... ]

        # extract the names of all the bars
        $BarNames = [];
        foreach ($this->Data as $Entries)
        {
            foreach ($Entries as $BarName => $YVal)
            {
                $BarNames[$BarName] = 1;
            }
        }
        $BarNames = array_keys($BarNames);

        # start the chart off with no data
        $this->Chart["data"]["columns"] = [];

        if ($this->AxisType == self::AXIS_CATEGORY)
        {
            # for categorical plots, data stays in place
            $Data = $this->Data;

            # set up X labels
            if ($this->ShowCategoryLabels)
            {
                $this->Chart["axis"]["x"]["categories"] =
                    $this->ShortCategoryNames($BarNames);
            }
            else
            {
                $this->Chart["axis"]["x"]["categories"] =
                    array_fill(0, count($BarNames), "");
            }

            # and fix up the display for single-category charts
            if ($this->SingleCategory)
            {
                $this->Chart["tooltip"]["grouped"] = FALSE;
            }
        }
        else
        {
            # for time series data, we need to sort our data into bins
            $Data = $this->SortDataIntoBins($BarNames);

            # convert our timestamps to JS-friendly date strings
            $Timestamps = array_keys($Data);
            array_walk($Timestamps, function(&$Val, $Key)
            {
                $Val = strftime("%Y-%m-%d", $Val);
            });
            array_unshift($Timestamps, "x-timestamp-x");

            # add this in to our data columns
            $this->Chart["data"]["columns"][]= $Timestamps;
        }

        # generate one row of data per bar to use for plotting

        # see http://c3js.org/reference.html#data-columns for format of 'columns' element.
        # since C3 always uses the label in 'columns' for the legend,
        # we'll need to populate the TooltipLabels array that is keyed
        # by legend label where values give the tooltip label
        foreach ($BarNames as $BarName)
        {
            $Label = isset($this->Labels[$BarName]) ?
                $this->Labels[$BarName] : $BarName ;

            if (isset($this->LegendLabels[$BarName]))
            {
                $MyLabel = $this->LegendLabels[$BarName];
                $this->TooltipLabels[$MyLabel] = $Label;
            }
            else
            {
                $MyLabel = $Label;
            }

            $DataRow = [$MyLabel];
            foreach ($Data as $Entries)
            {
                $DataRow[]= isset($Entries[$BarName]) ? $Entries[$BarName] : 0;
            }
            $this->Chart["data"]["columns"][] = $DataRow;
        }

        $this->Chart["data"]["type"] = "bar";

        if ($this->AxisType == self::AXIS_CATEGORY)
        {
            $this->Chart["axis"]["x"]["type"] = "category";
        }
        else
        {
            $this->AddToChart([
                "data" => [
                    "x" => "x-timestamp-x",
                    "xFormat" => "%Y-%m-%d",
                ],
                "axis" => [
                    "x" => [
                        "type" => "timeseries",
                    ],
                ],
            ]);
        }

        if (!is_null($this->BarWidth))
        {
            $this->Chart["bar"]["width"]["ratio"] = $this->BarWidth / 100;
        }

        if (!is_null($this->YLabel))
        {
            $this->Chart["axis"]["y"]["label"] = $this->YLabel;
        }

        if ($this->Zoom)
        {
            $this->Chart["zoom"]["enabled"] = TRUE;
        }

        if ($this->Stacked)
        {
            $this->Chart["data"]["groups"] = [
                $this->ShortCategoryNames($BarNames),
            ];
        }

        if ($this->Horizontal)
        {
            $this->Chart["axis"]["rotated"] = TRUE;
        }

        if ($this->Gridlines)
        {
            $this->Chart["grid"]["y"]["show"] = TRUE;
        }
    }

    /**
    * Sort the user-provided data into bins with sizes given by
    * $this->AxisType, filling in any gaps in the data given by the
    * user.
    * @param array $BarNames Bars that all bins should have.
    * @return Array of binned data.
    */
    protected function SortDataIntoBins($BarNames)
    {
        # create an array to store the binned data
        $BinnedData = [];

        # iterate over all our input data.
        foreach ($this->Data as $TS => $Entries)
        {
            # place this timestamp in the appropriate bin
            $TS = $this->BinTimestamp($TS);

            # if we have no results in this bin, then these are
            # the first
            if (!isset($BinnedData[$TS]))
            {
                $BinnedData[$TS] = $Entries;
            }
            else
            {
                # otherwise, iterate over the keys we were given
                foreach ($Entries as $Key => $Val)
                {
                    # if we have a value for this key
                    if (isset($BinnedData[$TS][$Key]))
                    {
                        # then add this new value to it
                        $BinnedData[$TS][$Key] += $Val;
                    }
                    else
                    {
                        # otherwise, insert the new value
                        $BinnedData[$TS][$Key] = $Val;
                    }
                }
            }
        }

        ksort($BinnedData);
        reset($BinnedData);

        # build up a revised data set with no gaps
        $GaplessData = [];
        # prime the revised set with the first element
        $GaplessData[key($BinnedData)] = current($BinnedData);

        # iterate over the remaining elements
        while (($Row = next($BinnedData)) !== FALSE)
        {
            $BinsAdded = 0;

            # if the next element is not the next bin, add an empty element
            while (key($BinnedData) != $this->NextBin(key($GaplessData)))
            {
                $GaplessData[$this->NextBin(key($GaplessData))] =
                    array_fill_keys($BarNames, 0);
                end($GaplessData);

                if ($BinsAdded > 1000)
                {
                    throw new Exception(
                        "Over 1000 empty bins added. "
                        ."Terminating possible infinite loop.");
                }
            }

            # and add the current element
            $GaplessData[key($BinnedData)] = $Row;
            end($GaplessData);
        }

        return $GaplessData;
    }

    /**
    * Determine which bin a specified timestamp belongs in.
    * @param mixed $TS Input timestamp.
    * @return int UNIX timestamp for the left edge of the bin.
    */
    protected function BinTimestamp($TS)
    {
        if (!preg_match("/^[0-9]+$/", $TS))
        {
            $TS = strtotime($TS);
        }

        switch ($this->AxisType)
        {
            case self::AXIS_TIME_DAILY:
                return strtotime(strftime("%Y-%m-%d 00:00:00", $TS));
                break;

            case self::AXIS_TIME_WEEKLY:
                $DateInfo = strptime(strftime(
                    "%Y-%m-%d 00:00:00", $TS), "%Y-%m-%d %H:%M:%S");

                $Year = $DateInfo["tm_year"] + 1900;
                $Month = $DateInfo["tm_mon"] + 1;
                $Day = $DateInfo["tm_mday"] - $DateInfo["tm_wday"];

                return mktime(0, 0, 0, $Month, $Day, $Year);
                break;

            case self::AXIS_TIME_MONTHLY:
                return strtotime(strftime("%Y-%m-01 00:00:00", $TS));
                break;

            case self::AXIS_TIME_YEARLY:
                return strtotime(strftime("%Y-01-01 00:00:00", $TS));
                break;
        }
    }

    /**
    * Get the next bin.
    * @param int $BinTS UNIX timestamp for the left edge of the current bin.
    * @return int UNIX timestamp for the left edge of the next bin.
    */
    protected function NextBin($BinTS)
    {
        $ThisBin = strftime("%Y-%m-%d %H:%M:%S", $BinTS);
        $Units = [
            self::AXIS_TIME_DAILY => "day",
            self::AXIS_TIME_WEEKLY => "week",
            self::AXIS_TIME_MONTHLY => "month",
            self::AXIS_TIME_YEARLY => "year",
        ];

        return strtotime($ThisBin." + 1 ".$Units[$this->AxisType]);
    }

    /**
    * Get abbreviated category names (e.g., for the legend).
    * @param array $LongNames Array of data keyed by long category names.
    * @return array of possibly abbreviated category names.
    */
    protected function ShortCategoryNames($LongNames)
    {
        $ShortNames = [];

        foreach ($LongNames as $Name)
        {
            $ShortNames[]= isset($this->LegendLabels[$Name]) ?
                $this->LegendLabels[$Name] : $Name ;
        }

        return $ShortNames;
    }

    protected $AxisType = self::AXIS_CATEGORY;
    protected $YLabel = NULL;
    protected $Zoom = FALSE;
    protected $Stacked = FALSE;
    protected $Horizontal = FALSE;
    protected $SingleCategory = FALSE;
    protected $Gridlines = TRUE;
    protected $ShowCategoryLabels = TRUE;
    protected $BarWidth = NULL;
}
